package clientPkg;

import interfacePkg.*;

import javax.swing.*;

import clientPkg.Waiting.WaitThread;

import java.awt.*;
import java.awt.event.*;

public class Board<O extends SinglePlayerServantIFC> implements Constants, MouseListener, MouseMotionListener {
	private Attempt[] attempts;
	private Attempt hiddenCode;
	private int currentAttempt;
	private JFrame boardFrame;
	private Point location;
	private MouseEvent pressed;
	private int numAttempts;
	private int numBalls;
	private int numColors;
	private int boardOwner;
	private O mmobj;
	private Board<O> opponentBoard;
	private int gameType;
	private ChatBox<O> chatFrame;
	private int boardWidth;
	private int boardHeight;
	private String playerName;
	private String opponentName;
	private EndThread endThread;
	private boolean waiting;
	
	//Creates the game boards
	public Board(int gt, int na, int nb, int nc, final O mo, String pn) {
		gameType = gt;
		boardOwner = PLAYERBOARD;
		mmobj = mo;
		numAttempts = na;
		numBalls = nb;
		numColors = nc;
		playerName = pn;
		
		if(gameType==MULTIPLAYER) {
			//Multiplayer
			MultiPlayerServantIFC multimmobj = (MultiPlayerServantIFC) mmobj;
			
			try {
				//Get opponent name
				opponentName = multimmobj.getOpponentName(playerName);
			} catch (Exception e) {
				e.printStackTrace();
			}
			
			//Create the board
			createBoard();
			
			//Create the opponents board
			opponentBoard = new Board<O>(numAttempts, numBalls, numColors, opponentName, mmobj);
			
			//Create the chatFrame
			chatFrame = new ChatBox<O>(mmobj, boardWidth, boardHeight, playerName);
			
			//Always check if the opponent has ended their game
			endThread = new EndThread(boardFrame, multimmobj);
			endThread.start();
		} else {
			//Create the board
			createBoard();
		}
	}
	
	//Creates an opponent board
	private Board(int na, int nb, int nc, String pn, O mo) {
		boardOwner = OPPONENTBOARD;
		numAttempts = na;
		numBalls = nb;
		numColors = nc;
		opponentName = pn;
		mmobj = mo;
		
		//Create the board
		createBoard();
	}
    
	public void createBoard() {	
		attempts = new Attempt[numAttempts];
    	currentAttempt = 0;
    	
    	//Create ColorSelector
    	ColorSelector colorSelector = new ColorSelector();
    	
    	//Setup the top level frame (window)
		boardFrame = new JFrame("Mastermind");
		
		//Add mouse listeners
		boardFrame.addMouseListener(this);
		boardFrame.addMouseMotionListener(this);
		
		//Determine the size the board should be
		boardHeight = ((numAttempts*45)+50)+75;
		if(boardHeight<MINBOARDHEIGHT) {
			boardHeight = MINBOARDHEIGHT;
		}
		
		boardWidth = BOARDWIDTH+(numBalls*(BALLWIDTH+5));
		
		//Set the size of the frame
		boardFrame.setPreferredSize(new Dimension(boardWidth, boardHeight));
		
		//Set the location of the frame
    	boardFrame.setLocation(BOARDSTARTX,BOARDSTARTY);
    	
    	//If this is the opponent's board move it beside the player's board
    	if(boardOwner==OPPONENTBOARD){
    		boardFrame.setLocation(BOARDSTARTX+(BOARDWIDTH+(numBalls*BALLWIDTH)),BOARDSTARTY);
    	}

		//Create the boardPanel
		JPanel boardPanel = new JPanel();
		boardPanel.setLayout(new GridBagLayout());
		boardPanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
		GridBagConstraints constraints = new GridBagConstraints();
		boardFrame.getContentPane().add(boardPanel);
		
		//Create panel to hold masterCode
		JPanel masterCodePanel = new JPanel();
		constraints.gridx = 0;
		constraints.gridy = 0;
		masterCodePanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
		boardPanel.add(masterCodePanel, constraints);
		
		//Hiddent code attempt
		hiddenCode = new Attempt(colorSelector,numBalls,-1);
		masterCodePanel.add(hiddenCode);
		
		//Create panel to hold attempts
		JPanel attemptPanel = new JPanel();
		constraints.gridx = 0;
		constraints.gridy = 1;
		attemptPanel.setLayout(new BoxLayout(attemptPanel, BoxLayout.PAGE_AXIS));
		boardPanel.add(attemptPanel, constraints);
		
		//Create the attempt panels
    	for(int i=numAttempts;0<i;i--) {		    		
    		Attempt a = new Attempt(colorSelector, numBalls, i);
    		
    		//Check if this is the first attempt and the players board
    		if(i==1 && boardOwner==PLAYERBOARD) {
    			//Unlock the first attempt's balls
    			a.unlockBalls();
    		}
    		
    		attempts[i-1] = a;
	    	attemptPanel.add(a);
    	}
		
		//Create the Banner Panel
    	JPanel bannerPanel = new JPanel();
    	constraints.gridheight = 3;
    	constraints.gridx = 1;
		constraints.gridy = 0;
		bannerPanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
    	boardPanel.add(bannerPanel, constraints);
    	
    	final JLabel bannerLabel;
    	
    	//Create the game banner label
    	if(boardOwner==PLAYERBOARD && gameType==SINGLEPLAYER) {
        	bannerLabel = new JLabel("<html><h3><center>M<br>A<br>S<br>T<br>E<br>R<br>M<br>I<br>N<br>D</center></h3></html>");
    	} else if(boardOwner==PLAYERBOARD) {
    		//Convert playerName to HTML Output
    		String playerNameHTML = "";
    		for(int i=0; i<playerName.length();i++) {
    			playerNameHTML += playerName.charAt(i);
    			playerNameHTML += "<br>";
    		}
    		
    		bannerLabel = new JLabel("<html><h3><center>"+playerNameHTML+"</center></h3></html>");
    	} else {
    		//Convert opponentName to HTML Output
    		String opponentNameHTML = "";
    		
    		for(int i=0; i<opponentName.length();i++) {
    			opponentNameHTML += opponentName.charAt(i);
    			opponentNameHTML += "<br>";
    		}
    		
    		bannerLabel = new JLabel("<html><h3><center>"+opponentNameHTML+"</center></h3></html>");
    	}
    	
    	//Add the banner to the bannerPanel
    	bannerPanel.add(bannerLabel);
    	
    	
    	if(boardOwner==PLAYERBOARD) {
	    	//Create the SelectPanel
	    	SelectPanel selectPanel = new SelectPanel(colorSelector, numColors);
	    	constraints.gridheight = 1;
	    	constraints.gridx = 0;
			constraints.gridy = 2;
			constraints.ipady = 25;
			selectPanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
	    	boardPanel.add(selectPanel, constraints);
	    	constraints.ipady = 1;
	    	
	    	//Create the ButtonsPanel
	    	JPanel buttonsPanel = new JPanel();
	    	constraints.gridheight = 1;
	    	constraints.gridx = 0;
			constraints.gridy = 3;
			buttonsPanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
	    	boardPanel.add(buttonsPanel, constraints);
	    	
	    	//Create the clearButton
			JButton clearButton = new JButton("Clear");
			constraints.weighty = 0;
			constraints.gridwidth = 1;
			constraints.gridx = 0;
			constraints.gridy = 1;
			clearButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					attempts[currentAttempt].clearBalls();
				}
			});
			buttonsPanel.add(clearButton, constraints);
	    	
	    	//Create the submit button
			JButton submitButton = new JButton("Submit");
			//constraints.weighty = 0;
			constraints.gridwidth = 1;
			constraints.gridx = 0;
			constraints.gridy = 3;
			submitButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					boolean invalid = false;
					int colors[] = new int[numBalls];
					
					//Get the values of the balls and submit to the server
					colors = attempts[currentAttempt].getBallColors();
					
					//Check for empty spot
					for(int i=0; i<numBalls; i++) {
						if(colors[i]==0) {
							invalid = true;
						}
					}
					
					if(invalid!=true) {
						//Lock the currentAttempt balls
						attempts[currentAttempt].lockBalls();
						
						int pegs[] = new int[numBalls];
						try {
							if(gameType==SINGLEPLAYER) {
								//Send the chosen colors to the remote object
								//mmobj.submitAttempt(input);
								mmobj.submitAttempt(colors);
								
								//Receive the results from the remote object
								pegs = mmobj.receiveResult();
							
								//Send the array of pegs (result from server) to gui
								attempts[currentAttempt].setPegs(pegs);
								
								//Increase currentAttempt by one
								currentAttempt++;
								
								//Check if win
								if(mmobj.checkEndGame()==WIN){
									int[] balls = mmobj.getMasterCode();
									hiddenCode.setBalls(balls);
									JOptionPane.showMessageDialog(boardFrame, "You Win!!", "Winner", JOptionPane.PLAIN_MESSAGE);
									new MainMenu();
									boardFrame.setVisible (false);
									boardFrame.dispose();
								} else if(mmobj.checkEndGame()==LOSE){
									int[] balls = mmobj.getMasterCode();
									hiddenCode.setBalls(balls);
									JOptionPane.showMessageDialog(boardFrame, "You Lost!", "Loser", JOptionPane.ERROR_MESSAGE);
									new MainMenu();
									boardFrame.setVisible (false);
									boardFrame.dispose();
								} else {
									//Unlock next attempt balls
									attempts[currentAttempt].unlockBalls();
								}
							} else {
								//Run submit thread which waits for results from the server
								SubmitThread st = new SubmitThread(boardFrame, colors);
								st.start();
							}
						} catch (Exception ex) {
							ex.printStackTrace();
							JOptionPane.showMessageDialog(boardFrame, "Error: Connection to server broken!\n-Check your network connection.\n-Server may be down.", "AHH!", JOptionPane.ERROR_MESSAGE);
						}
					} else {
						JOptionPane.showMessageDialog(boardFrame, "All the spots for your current attempt must be filled in!", "zOMG!1", JOptionPane.ERROR_MESSAGE);
					}
				}
			});
			buttonsPanel.add(submitButton, constraints);
			
	    	//Create the Menu
	    	BoardMenu<O> bm = new BoardMenu<O>(this, gameType, mmobj, boardFrame, playerName);
	    	boardFrame.setJMenuBar(bm);
    	} else {
    		//Create the ButtonsPanel
	    	JPanel buttonsPanel = new JPanel();
	    	constraints.gridheight = 1;
	    	constraints.gridx = 0;
			constraints.gridy = 2;
			buttonsPanel.setLayout(new GridBagLayout());
			buttonsPanel.setBackground(new Color(RCOLOR, GCOLOR, BCOLOR));
			buttonsPanel.setPreferredSize(new Dimension(200,100));
	    	boardPanel.add(buttonsPanel, constraints);
	    	
	    	GridBagConstraints buttonsConstraints = new GridBagConstraints();
	    	buttonsConstraints.insets = new Insets(5,5,5,5);
	    	
	    	//Ability label
			JLabel abilityLabel = new JLabel("<html><h3>Mind-Reader</h3></html>");
			abilityLabel.setPreferredSize(new Dimension(200,100));
			buttonsConstraints.gridx = 0;
			buttonsConstraints.gridy = 1;
			buttonsConstraints.gridwidth = 2;
			buttonsPanel.add(abilityLabel, buttonsConstraints);
	    	
	    	//Number of attempts ComboBox
			final JComboBox abilityComboBox = new JComboBox();
			
			//Create the combobox items
			for(int i=0;i<numAttempts;i++) {
				abilityComboBox.addItem(i+1);
			}
			
			buttonsConstraints.gridwidth = 1;
			buttonsConstraints.gridx = 0;
			buttonsConstraints.gridy = 2;
			abilityComboBox.setPreferredSize(new Dimension(70,25));
			buttonsPanel.add(abilityComboBox, buttonsConstraints);
	    	
	    	//Create the 6thSenseButton
			final JButton abilityButton = new JButton("Reveal Attempt");
			buttonsConstraints.weighty = 0;
			buttonsConstraints.gridwidth = 1;
			buttonsConstraints.gridx = 1;
			buttonsConstraints.gridy = 2;
			abilityButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					//Get the selected value from the combobox
					int selectedAttempt = Integer.parseInt(abilityComboBox.getSelectedItem().toString())-1;
					
					//Check if the attempt number has been completed
					if(selectedAttempt<currentAttempt) {
						//Multiplayer
						MultiPlayerServantIFC multimmobj = (MultiPlayerServantIFC) mmobj;
						
						//Send the selected attempted number to the servant and get the response
						try {
							int[] balls = multimmobj.revealAttempt(opponentName, selectedAttempt);

							//Update the opponents gui balls
							attempts[selectedAttempt].setBalls(balls);
							
							//Disable button and combobox
							abilityButton.setEnabled(false);
							abilityComboBox.setEnabled(false);
						} catch (Exception ex) {
							ex.printStackTrace();
						}
					} else {
						//Else show message invalid attempt chosen
						JOptionPane.showMessageDialog(boardFrame, "You may only choose an attempt that has been completed!", "zOMG!1", JOptionPane.ERROR_MESSAGE);
					}
				}
			});
			buttonsPanel.add(abilityButton, buttonsConstraints);

			//Waiting label
			JLabel waitingLabel = new JLabel("Waiting");
			waitingLabel.setPreferredSize(new Dimension(200,200));
			buttonsConstraints.gridx = 0;
			buttonsConstraints.gridy = 3;
			buttonsConstraints.gridwidth = 2;
			buttonsConstraints.ipady = 10;
			buttonsPanel.add(waitingLabel, buttonsConstraints);
			buttonsConstraints.ipady = 0;
			
			//Start WaitingThread
    		WaitingThread wt = new WaitingThread(waitingLabel, opponentName);
    		wt.start();
			
    		//Display the empty menu
    		JMenuBar opponentMenuBar = new JMenuBar();
    		JMenu aboutMenu = new JMenu("?");
    		aboutMenu.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					JOptionPane.showMessageDialog(boardFrame,"Programmed by Warren Brown & Brian Thai","ALERT",JOptionPane.NO_OPTION);
				}
			});
    		opponentMenuBar.add(aboutMenu);
    		boardFrame.setJMenuBar(opponentMenuBar);
    	}
    	
    	//Borderless window
    	boardFrame.setUndecorated(true);
    	
		//Resize the window
    	boardFrame.pack();
		
		//Display the window
    	boardFrame.setVisible(true);
    }

	public void setOpponentPegs(int[] pegs) {
		//Send the array of playerPegs (result from server) to gui
		opponentBoard.attempts[currentAttempt].setPegs(pegs);
	}
	
	public void closeBoard() {
		boardFrame.setVisible (false);
		boardFrame.dispose();
	}
	
	public void closeGame() {
		try {
			//Close the EndThread
			endThread.closeThread();
			
			//Close the opponent board
			opponentBoard.closeBoard();
			
			//Close the chat window
			chatFrame.close();
		} catch (Exception e) {}
		
		//Close the player board
		closeBoard();
	}
	
	//Waiting Thread Class -Used to check if the other player is waiting or not
	class WaitingThread extends Thread {
		private JLabel waitingLabel;
		private String opponentName;
		private boolean loop;
		
		public WaitingThread(JLabel wl, String op) {
			opponentName = op;
			waitingLabel = wl;
			loop = true;
		}
		
		public void closeThread() {
			loop = false;
		}

		public void run() {
			try {
				//Multiplayer
				MultiPlayerServantIFC multimmobj = (MultiPlayerServantIFC) mmobj;
				
				while(loop) {
					//Get the value of the player status
					if(multimmobj.getPlayerStatus(opponentName)==PLAYERWAITING) {
						//Update the label on the board
						waitingLabel.setText("Waiting For You...");
					} else {
						waitingLabel.setText(" ");
					}
					Thread.sleep(100);
				}
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
	}
	
	//Submit Thread Class -Used to run the proccess of sending attempt to server and waiting for results
	class SubmitThread extends Thread {
		private JFrame boardFrame;
		private boolean loop;
		private int[] colors;
		
		public SubmitThread(JFrame bf, int[] c) {
			boardFrame = bf;
			loop = true;
			colors = c;
		}
		
		public void closeThread() {
			loop = false;
		}

		public void run() {
			if(waiting==false) {
				waiting=true;
				try {
					//Multiplayer
					MultiPlayerServantIFC multimmobj = (MultiPlayerServantIFC) mmobj;
					
					//Send the chosen colors to the remote object
					multimmobj.submitAttempt(playerName, currentAttempt, colors);
					
					//Check receive is ready
					while(multimmobj.checkReceive()==false) {}
					
					//Get player results from the server
					int[] playerPegs = multimmobj.receiveResult(playerName);
					
					//Get opponent results from the server
					int[] opponentPegs = multimmobj.receiveResult(opponentName);
					
					//Send the array of playerPegs (result from server) to gui
					attempts[currentAttempt].setPegs(playerPegs);
					
					//Send the array of opponentPegs gui
					setOpponentPegs(opponentPegs);
					
					//Check if win
					if(multimmobj.checkEndGame(playerName, currentAttempt)==WIN){
						int[] balls = multimmobj.getMasterCode();
						hiddenCode.setBalls(balls);
						JOptionPane.showMessageDialog(boardFrame, "You Win!!", "Winner", JOptionPane.PLAIN_MESSAGE);
						new MainMenu();
						closeGame();
						multimmobj.removeGame();
					} else if(multimmobj.checkEndGame(playerName, currentAttempt)==LOSE){
						int[] balls = multimmobj.getMasterCode();
						hiddenCode.setBalls(balls);
						JOptionPane.showMessageDialog(boardFrame, "You Lose!", "Loser", JOptionPane.ERROR_MESSAGE);
						new MainMenu();
						closeGame();
						multimmobj.removeGame();
					} else if(multimmobj.checkEndGame(playerName, currentAttempt)==TIE){
						int[] balls = multimmobj.getMasterCode();
						hiddenCode.setBalls(balls);
						JOptionPane.showMessageDialog(boardFrame, "Tie Game!", "Tie", JOptionPane.ERROR_MESSAGE);
						new MainMenu();
						closeGame();
						multimmobj.removeGame();
					} else {
						//Increase currentAttempt by one
						currentAttempt++;
						opponentBoard.currentAttempt++;
						
						//Unlock next attempt balls
						attempts[currentAttempt].unlockBalls();
					}
				} catch (Exception ex) {
					ex.printStackTrace();
				}
				waiting=false;
				closeThread();
			}
		}
	}
	
	//End Thread Class -Used to run an infinate loop until checking if the opponent has quit
	class EndThread extends Thread {
		MultiPlayerServantIFC mmobj;
		private JFrame boardFrame;
		private boolean loop;
		
		public EndThread(JFrame bf, MultiPlayerServantIFC mo) {
			mmobj = mo;
			boardFrame = bf;
			loop = true;
		}
		
		public void closeThread() {
			loop = false;
		}

		public void run() {
			while(loop) {
	    		try {
	        		//Check if game is ready to start
	    			if(!mmobj.checkExit(playerName)) {
	    				//If the opponent quits their game show message that the player has quit
	    				JOptionPane.showMessageDialog(boardFrame, "Your opponent left!", "AHH!", JOptionPane.ERROR_MESSAGE);
	    				
	    				//Close the game
	    				closeGame();
	    				
	    				//Remove the game
						mmobj.removeGame();
						
						//Display Main Menu
						new MainMenu();
						
	    				break;
	    			}
	    			Thread.sleep(100);
	    		} catch(Exception e) {
	    			e.printStackTrace();
	    		}
			}
			closeThread();
		}
	}
	
	public void mousePressed(MouseEvent me) {
		pressed = me;
	}
 
	public void mouseClicked(MouseEvent e) {}
	public void mouseReleased(MouseEvent e) {}
 
	public void mouseDragged(MouseEvent me) {
		location = boardFrame.getLocation(location);
		int x = location.x - pressed.getX() + me.getX();
		int y = location.y - pressed.getY() + me.getY();
		boardFrame.setLocation(x, y);
	}
 
	public void mouseMoved(MouseEvent e) {}
	public void mouseEntered(MouseEvent e) {}
	public void mouseExited(MouseEvent e) {}
}
